<?php
/**
 * Merge clover.xml and junit.xml from several phpunit reports.
 * Parameters:
 * --in inputfile.xml
 * --out outputfile.xml (optional: default STDOUT)
 * --report <clover|junit> (optional: default is junit)
 * 
 * Sample calls:
 * php merge_phpunit_reports.php --in clover1.xml --in clover2.xml --out result2.xml --report clover
 * php merge_phpunit_reports.php --in junit1.xml --in junit2.xml
 * 
 * Tested with report files from phpunit 3.7.28
 */

$inputFiles = array();
$outputFile = '';
$reportType = 'junit';
$allowedTypes = array('junit', 'clover');

$args = $_SERVER['argv'];
array_shift($args);
while ($cmd = array_shift($args)) {
    if ($cmd == '--in') {
        $file = array_shift($args);
        if (! file_exists($file)) {
            echo sprintf("Error: input file %s does not exist.\n", $file);
            exit(1);
        }
        $inputFiles[] = $file;
        continue;
    }
    if ($cmd == '--out') {
        $outputFile = array_shift($args);
        if ($outputFile === null) {
            echo "Error: missing argument for output file.\n";
            exit(2);
        } 
    }
    if ($cmd == '--report') {
        $reportType = array_shift($args);
        if ($reportType === null) {
            echo "Error: missing argument for report type.\n";
            exit(3);
        }
        if (!in_array($reportType, $allowedTypes)) {
            echo sprintf("Error: invalid report type %s.\n", $reportType);
            exit(4);
        }
    }
}
if (count($inputFiles) < 2) {
    echo "Error: to merge reports we need at least two input files.\n";
    exit(5);
}

$xml1 = getDomDocument($inputFiles[0]);
for ($i = 1; $i < count($inputFiles); $i++) {
    $xml2 = getDomDocument($inputFiles[$i]);
    if ($reportType == 'clover') {
        mergeClover($xml1, $xml2);
    }
    elseif ($reportType == 'junit') {
        mergeJunit($xml1, $xml2);
    }
}

if (mb_strlen($outputFile) > 0) {
    if (! $xml1->save($outputFile)) {
        echo "Error: write XML to file $fileout\n";
        exit(6);
    }
} else {
    echo $xml1->saveXML();
}

function mergeJunit(&$xml1, &$xml2)
{
    $numericAttrs = array('tests', 'assertions', 'failures', 'errors', 'time');
    $xpath2 = new DOMXPath($xml2);
    $res2 = $xpath2->query('/testsuites/testsuite/testsuite');
    if (!is_null($res2)) {
        foreach ($res2 as $testsuite) {
            if (!$testsuite->hasChildNodes()) {
                continue;
            }
            $srcTs = getAttributes($testsuite);
            if (isset($srcTs['name']) && $srcTs['name'] != '') {
                $xpath1 = new DOMXPath($xml1);
                $res1 = $xpath1->query(sprintf('//testsuite[@name="%s"]', $srcTs['name']));
                // if the same testsuite does not exist in xml1 then just copy the whole
                // testsuite from xml2 into xml1
                if (is_null($res1)) {
                    $new = $xml1->importNode($testsuite, true);
                    $res = $xpath1->query('/testsuites');
                    if (!is_null($res)) {
                        $res->item(0)->appendChild($new);
                    }
                }
                // we have a testsuite at xml1 that has the same name
                // so add all child nodes from the current testsuite in xml2 into
                // the same testsuite in xml1
                else {
                    if ($res1->item(0)->hasChildNodes()) {
                        $tgTs = getAttributes($res1->item(0));
                    }
                    // merge attributes of both testsuite nodes
                    foreach($srcTs as $key => $value) {
                        if (isset($tgTs[$key])) {
                            if (in_array($key, $numericAttrs)) {
                                $tgTs[$key] += $srcTs[$key];
                            }                        
                        } else {
                            $tgTs[$key] = $value;
                        }
                    }
                    // change the attributes of the testsuite in xml1
                    modifyAttributes($xml1, $res1->item(0), $tgTs);
                    // append all child elements from testsuite in xml2 to the target testsuite in xml1
                    foreach ($testsuite->childNodes as $node) {
                        if ($node->nodeType == XML_ELEMENT_NODE) {
                            $new = $xml1->importNode($node, true);
                            $res1->item(0)->appendChild($new);
                        }
                    }
                } 
            }
        }
    }
    // correct numbers in main testsuite attributes (the sum of all sub nodes testsuites
    $globalSuite = $xpath1->query('/testsuites/testsuite');
    $globalAttr = getAttributes($globalSuite->item(0));
    // reset the numbers
    foreach ($globalAttr as $key => $value) {
        if (in_array($key, $numericAttrs)) {
            $globalAttr[$key] = 0;
        }
    }
    // now select all child testsuites (second level) and count up numbers
    foreach ($globalSuite->item(0)->childNodes as $node) {
        if ($node->nodeType == XML_ELEMENT_NODE && $node->nodeName == 'testsuite') {
            if ($node->hasAttributes()) {
                foreach ($node->attributes as $attr) {
                    if (in_array($attr->nodeName, $numericAttrs)) {
                        if (isset($globalAttr[$attr->nodeName])) {
                            $globalAttr[$attr->nodeName] += $attr->nodeValue;
                        } else {
                            $globalAttr[$attr->nodeName] = $attr->nodeValue;
                        }
                    }
                }
            }
        }
    }
    modifyAttributes($xml1, $globalSuite->item(0), $globalAttr);
}

function mergeClover(&$xml1, &$xml2)
{
    $files = $xml2->getElementsByTagName('file');
    $project = $xml1->getElementsByTagName('project')->item(0);
    for($i=0; $i < $files->length; $i++) {
        $item = $files->item($i);
        $new  = $xml1->importNode($item, true); 
        $project->appendChild($new);
    }
    
    $newMetrics = array();

    $xpath1 = new DOMXPath($xml1);
    $res1 = $xpath1->query('/coverage/project/metrics');
    if (!is_null($res1)) {
        $metrics1 = $res1->item(0);
        $newMetrics = getAttributes($metrics1);
    }
    $xpath2 = new DOMXPath($xml2);
    $res2 = $xpath2->query('/coverage/project/metrics');
    if (!is_null($res2)) {
        $metrics2 = $res2->item(0);
        $metrics2Attr = getAttributes($metrics2);
    }
    // merge metrics attribute arrays by modifying the new metrics node
    foreach (array_keys($newMetrics) as $key) {
        if (isset($metrics2Attr[$key])) {
            $newMetrics[$key] += $metrics2Attr[$key];
            unset($metrics2Attr[$key]);
        }
    }
    foreach ($metrics2 as $key => $value) {
        $newMetrics[$key] = $value;
    }
    $node = $xml1->createElement('metrics');
    modifyAttributes($xml1, $node, $newMetrics);
    // add metrics node to document right below element project
    $project = $xml1->getElementsByTagName('project')->item(0);
    //$project->replaceChild($node, $metrics1);
    $project->removeChild($metrics1);
    $project->appendChild($node);
}

function modifyAttributes($doc, $node, $attributes)
{
    // change the attributes of the testsuite in xml1
    // walk trough all existin attributes and change them to the new value
    foreach($node->attributes as $attr) {
        if (!isset($attributes[$attr->nodeName])) {
            continue;
        }
        $attr->nodeValue = $attributes[$attr->nodeName];
        unset($attributes[$attr->nodeName]);
    }
    // any attributes that may not yet exist need to be created
    foreach($attributes as $key => $value) {
        $attr = $doc->createAttribute($key);
        $attr->value = $value;
        $node->appendChild($attr);
    }
}

function getAttributes($node)
{
    $res = array();
    if ($node->hasAttributes()) {
        foreach ($node->attributes as $attr) {
            $res[$attr->nodeName] = $attr->nodeValue;
        }
    }
    return $res;
}

function getDomDocument($file)
{
    $doc = new DOMDocument();
    if (! $doc->load($file)) {
        echo "Error loading the input file $file\n";
        exit(7);
    }
    return $doc;
    
    $sxe = simplexml_load_file($file);
    if ($sxe === false) {
        echo "Error loading the input file $file\n";
        exit(7);
    }
    $dom_sxe = dom_import_simplexml($sxe);
    return $dom_sxe;
    if (! $dom_sxe) {
        echo "Error converting XML from file $file into DOM.\n";
        exit(8);
    }
    $dom = new DOMDocument('1.0');
    $dom_sxe = $dom->importNode($dom_sxe, true);
    $dom->appendChild($dom_sxe);
    
    return $dom;
}
